// dnaparser.js

/**
 * Initializes the DNAParse function by setting it as a global on the window object.
 *
 * This function should be called after the script has been loaded, such as with
 * a DOMContentLoaded event listener.
 */
function initDnaparser() {
    window.DNAParse = DNAParse;
}

/**
 * DNAParse parses a DNA string into an investor object with properties and influencers.
 *
 * The DNA string should be enclosed in curly braces and consists of semicolon-separated
 * key-value pairs. Influencer and PDDInfluencer sections are handled separately.
 *
 * @param {string} dna - The DNA string to parse.
 * @returns {object|null} An object containing investor properties and arrays of influencers
 *                        and PDD influencers, or null if the DNA string format is invalid.
 */
function DNAParse(dna) {
    let inv = {
        props: {},
        BirthCertificate: {},
        inf: [],
        pddinf: [],
        metricSet: new Set(),  // the set of metrics used by this investor's influencers
    };
    dna = dna.trim();
    if (dna[0] != '{' || dna[dna.length - 1] != '}') {
        console.log("Invalid DNA string format. DNA string must start with '{' and end with '}'.");
        return null;
    }
    dna = dna.slice(1, -1); // remove the surrounding braces
    let props = dna.split(';'); // split by semicolons

    //-----------------------------------------------------------------------------------
    // Make a dictionary of Investor Properties.  We ignore the "Investor" declaration.
    // We handle Influencer and PDDInfluencer separately.
    //-----------------------------------------------------------------------------------
    var inf = null;
    var pddinf = null;
    for (var i = 0; i < props.length; i++) {
        let prop = props[i].trim();
        if (!prop) continue;

        if (prop.startsWith("Influencer")) {
            inf = prop;
            continue;
        } else if (prop.startsWith("PDDInfluencer")) {
            pddinf = prop;
            continue;
        }

        // If not Influencer or PDDInfluencer and not "Investor"
        if (!prop.startsWith("Investor")) {
            let p = prop.split('=');
            let key = p[0].trim();
            let value = p.slice(1).join('=').trim();

            // Handle Options property
            if (key === "Options") {
                // value should look like {ID=...,FirstName=...,LastName=...}
                // Remove the leading '{' and trailing '}' from value
                if (value[0] === '{' && value[value.length - 1] === '}') {
                    value = value.slice(1, -1);
                }

                // Now value is something like "ID=2be0...,FirstName=John,LastName=Doe"
                let optionsArr = value.split(',');
                let optionsObj = {};
                for (let j = 0; j < optionsArr.length; j++) {
                    let optPair = optionsArr[j].split('=');
                    if (optPair.length == 2) {
                        let optKey = optPair[0].trim();
                        let optVal = optPair[1].trim();
                        optionsObj[optKey] = optVal;
                    }
                }
                inv.props.Options = optionsObj;
            } else if (key === "BirthCertificate") {
                // value is something like this: {DtStart=2023-12-16|DtStop=2024-12-16|Wcon=0.3333|Wcor=0.3333|Wprf=0.3333}
                // first, strip off the braces
                if (value[0] === '{' && value[value.length - 1] === '}') {
                    value = value.slice(1, -1);
                }
                // now we have something like: DtStart=2023-12-16|DtStop=2024-12-16|Wcon=0.3333|Wcor=0.3333|Wprf=0.3333
                // the vertical bar (|) is used as the delimiter as we will probably have values that contain lists,
                // for those we'll delimit with comma (,) -- this is the same pattern used in Influencers.
                let optionsArr = value.split('|');
                let birthCertificateObj = {};
                for (let j = 0; j < optionsArr.length; j++) {
                    let optPair = optionsArr[j].split('=');
                    if (optPair.length == 2) {
                        let optKey = optPair[0].trim();
                        let optVal = optPair[1].trim();
                        birthCertificateObj[optKey] = optVal;
                    }
                }
                inv.BirthCertificate = birthCertificateObj;
            } else {
                // Normal property
                inv.props[key] = value;
            }
        }
    }

    inv.inf = parseInfluencerList(inv, inf);
    inv.pddinf = parseInfluencerList(inv, pddinf);
    return inv;
}


/**
 * Parses a list of influencer DNA strings and returns an array of influencer objects.
 * Each influencer string is expected to be in the format:
 * "[{key1=value1,key2=value2,...}|{key1=value1,key2=value2,...}|...]".
 * The function extracts key-value pairs for each influencer, storing them in an object.
 * If a "Metric" key is found, its value is added to the 'metricSet' of the investor.
 *
 * @param {object} inv - The investor object containing a 'metricSet' property for metrics.
 * @param {string|null} inf - The string containing the influencer data, or null if absent.
 * @returns {Array<object>} An array of objects, each representing an influencer with properties.
 */
function parseInfluencerList(inv, inf) {
    let x = [];
    if (inf != null) {
        var idx = inf.indexOf("[");                     // PDDInfluencers=[{x=y,a=b}|{x=y,a=b}|{x=y,a=b}]
        var p = inf.slice(idx + 1, -1);                 // {x=y,a=b}|{x=y,a=b}|{x=y,a=b}
        var props = p.split('|');                       // split by vertical bars
        for (var i = 0; i < props.length; i++) {        // for each Influencer in the list
            let p = props[i].slice(1, -1);              // remove the outer braces {}
            let infprops = p.split(',');                // split by commas
            let infp = {};                              // declare the Influencer property dictionary
            for (var j = 0; j < infprops.length; j++) {
                let ip = infprops[j].split('=');        // split by equals
                if (ip.length == 2) {                   // if we have a key and a value
                    let key = ip[0].trim();             // assign the key
                    let value = ip[1].trim();           // assign the value
                    infp[key] = value;                  // assign the key and value
                    if (key == "Metric") {              // if the key is Metric...
                        inv.metricSet.add(value);       // ...add the metric to the set
                    }
                } else if (ip.length == 1) {
                    let key = ip[0].trim();             // assign the key
                    infp[key] = "";                     // at the moment, this is just "LSMInfluencer", there may be other influencer types in the future
                }
            }
            x.push(infp);                               // add the Influencer to the list
        }
    }
    return x
}

/**
 * Synchronous SHA-256 Hash Implementation in Pure JavaScript.
 * Returns a hexadecimal string of the hash.
 * Source: Public domain, lightweight implementation
 */
function sha256(ascii) {
    const K = [
        0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
        0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
        0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
        0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
        0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
        0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
        0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
        0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
    ];

    function rightRotate(value, amount) {
        return (value >>> amount) | (value << (32 - amount));
    }

    const words = [];
    let asciiBitLength = ascii.length * 8;

    let hash = [0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a, 0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19];

    // Preprocess the string into 512-bit chunks
    let i, j;
    for (i = 0; i < ascii.length; ++i) {
        words[i >> 2] |= ascii.charCodeAt(i) << (24 - (i % 4) * 8);
    }
    words[ascii.length >> 2] |= 0x80 << (24 - (ascii.length % 4) * 8);
    words[((ascii.length + 64 >> 9) << 4) + 15] = asciiBitLength;

    // Process each 512-bit chunk
    for (j = 0; j < words.length;) {
        const w = words.slice(j, (j += 16));
        const oldHash = hash.slice(0);

        for (i = 0; i < 64; ++i) {
            const w15 = w[i - 15],
                w2 = w[i - 2];
            w[i] =
                i < 16
                    ? w[i]
                    : (w[i - 16] +
                        (rightRotate(w15, 7) ^ rightRotate(w15, 18) ^ (w15 >>> 3)) +
                        w[i - 7] +
                        (rightRotate(w2, 17) ^ rightRotate(w2, 19) ^ (w2 >>> 10))) |
                    0;

            const a = hash[0],
                e = hash[4];
            const t1 =
                hash[7] +
                (rightRotate(e, 6) ^ rightRotate(e, 11) ^ rightRotate(e, 25)) +
                ((e & hash[5]) ^ (~e & hash[6])) +
                K[i] +
                w[i];
            const t2 = (rightRotate(a, 2) ^ rightRotate(a, 13) ^ rightRotate(a, 22)) + ((a & hash[1]) ^ (a & hash[2]) ^ (hash[1] & hash[2]));

            hash = [(t1 + t2) | 0].concat(hash);
            hash[4] = (hash[4] + t1) | 0;
        }

        for (i = 0; i < 8; ++i) hash[i] = (hash[i] + oldHash[i]) | 0;
    }

    return hash.map(h => ("00000000" + h.toString(16)).slice(-8)).join("");
}

/**
 * Reconstructs the DNA string from the parsed investor object, ensuring:
 * 1) "Investor" is always first;
 * 2) Top-level properties are in alphabetical order;
 * 3) Sub-objects (BirthCertificate, Options, etc.) are alphabetized;
 * 4) In each Influencer and PDDInfluencer, any property whose value === ""
 *    (e.g., "LSMInfluencer") goes first, followed by all other properties
 *    in alphabetical order.
 *
 * @param {object} inv - The parsed investor object from DNAParse().
 * @returns {string} The reconstructed DNA string.
 */
function GetDNAString(inv) {
    // "Investor" is not actually a property; it's a literal token that
    // we always place first.
    const segments = ["Investor"];

    // We'll collect all top-level properties (besides "Investor") in a map
    // so we can sort them alphabetically. That includes:
    //   - BirthCertificate  (object)
    //   - Options          (object)
    //   - C1, C2, Strategy, etc. (strings)
    //   - Influencers and PDDInfluencers (arrays of objects)
    const topProps = {};

    // 1) Add BirthCertificate if present
    if (inv.BirthCertificate && Object.keys(inv.BirthCertificate).length > 0) {
        topProps["BirthCertificate"] = inv.BirthCertificate;
    }

    // 2) Add everything in inv.props
    //    (we'll handle "Options" from here, too)
    for (const [key, val] of Object.entries(inv.props)) {
        topProps[key] = val;
    }

    // 3) Add Influencers / PDDInfluencers if present
    if (inv.inf && inv.inf.length > 0) {
        topProps["Influencers"] = inv.inf;
    }
    if (inv.pddinf && inv.pddinf.length > 0) {
        topProps["PDDInfluencers"] = inv.pddinf;
    }

    // Sort the top-level property names
    const sortedTopLevelKeys = Object.keys(topProps).sort((a, b) => a.localeCompare(b));

    // Build each top-level segment in alphabetical order
    for (const key of sortedTopLevelKeys) {
        const val = topProps[key];

        // --------------------------------
        // 1) BirthCertificate={...}
        // --------------------------------
        if (key === "BirthCertificate") {
            const bcKeys = Object.keys(val).sort((a, b) => a.localeCompare(b));
            const bcString = bcKeys.map(k => `${k}=${val[k]}`).join("|");
            segments.push(`BirthCertificate={${bcString}}`);
        }

        // --------------------------------
        // 2) Options={...}
        // --------------------------------
        else if (key === "Options") {
            if (val && typeof val === "object" && Object.keys(val).length > 0) {
                const optKeys = Object.keys(val).sort((a, b) => a.localeCompare(b));
                const optString = optKeys.map(k => `${k}=${val[k]}`).join(",");
                segments.push(`Options={${optString}}`);
            } else {
                segments.push(`Options={}`);
            }
        }

        // --------------------------------
        // 3) Influencers=[ ... ]
        // --------------------------------
        else if (key === "Influencers") {
            const infStrings = val.map(obj => buildInfluencerString(obj));
            segments.push(`Influencers=[${infStrings.join("|")}]`);
        }

        // --------------------------------
        // 4) PDDInfluencers=[ ... ]
        // --------------------------------
        else if (key === "PDDInfluencers") {
            const pddStrings = val.map(obj => buildInfluencerString(obj));
            segments.push(`PDDInfluencers=[${pddStrings.join("|")}]`);
        }

        // --------------------------------
        // 5) All other props (simple key=value)
        // --------------------------------
        else {
            segments.push(`${key}=${val}`);
        }
    }

    // Finally, join with semicolons and wrap in curly braces
    return `{${segments.join(";")}}`;
}

/**
 * Builds a single influencer string (either for Influencers or PDDInfluencers).
 * Ensures that any key with an empty value (e.g., "LSMInfluencer") is listed
 * first, then the rest sorted alphabetically.
 *
 * e.g. for an object like:
 *   {
 *       LSMInfluencer: "",
 *       BuySign: "true",
 *       Delta1: "-328",
 *       Delta2: "-49",
 *       Metric: "GCAM_C16_60"
 *   }
 *
 * we want a string like:
 *   "{LSMInfluencer,BuySign=true,Delta1=-328,Delta2=-49,Metric=GCAM_C16_60}"
 */
function buildInfluencerString(obj) {
    // Separate keys into two groups:
    // 1) Keys whose value === "" (to be placed first).
    // 2) Keys whose value !== "" (then sorted alphabetically).
    const allKeys = Object.keys(obj);
    const emptyValueKeys = allKeys.filter(k => obj[k] === "").sort((a, b) => a.localeCompare(b));
    const nonEmptyValueKeys = allKeys.filter(k => obj[k] !== "").sort((a, b) => a.localeCompare(b));

    // Now build the final array of property strings
    // First come the empty-value keys, with "key" only (no "=value").
    // Then come the normal key-value pairs in alphabetical order.
    const parts = [];

    // For each key that has an empty value, we append "key"
    for (const k of emptyValueKeys) {
        parts.push(k);
    }

    // For each key that has a non-empty value, we append "key=value"
    for (const k of nonEmptyValueKeys) {
        parts.push(`${k}=${obj[k]}`);
    }

    // Finally, wrap in curly braces with comma-separated properties
    // e.g. "{LSMInfluencer,BuySign=true,...}"
    return `{${parts.join(",")}}`;
}

/**
 * Computes the "core" DNA string for the given investor object,
 * using the same structure as the Go code’s GetDNACoreString().
 *
 * This core string includes (in alphabetical order of keys):
 *   - C1
 *   - C2
 *   - Influencers=[ {LSMInfluencer,BuySign=...,Delta1=...,Delta2=...,Metric=...} | ... ]
 *   - PDDInfluencers=[ {LSMInfluencer,BuySign=...,Delta1=...,Delta2=...,Metric=...} | ... ]
 *   - Strategy
 *
 * Each Influencer is also sorted in some *consistent* manner (the Go code
 * calls `SortInfluencers()` internally). Typically, you'd sort them by
 * e.g. `Metric` or something else. If you don’t know how they’re sorted in
 * Go, you can omit or do a simple alphabetical sort by the full influencer DNA.
 *
 * @param {object} inv - The parsed investor object from DNAParse().
 * @returns {string} The “core” DNA string used to compute the ID.
 */
function buildDNACoreString(inv) {
    // We’ll collect each core property in an array of strings, then
    // join them with semicolons at the end.
  
    // 1) Gather C1, C2, Strategy, Influencers, PDDInfluencers from inv.props
    const coreSegments = [];
  
    // C1
    if (inv.props.C1) {
      coreSegments.push(`C1=${inv.props.C1}`);
    }
  
    // C2
    if (inv.props.C2) {
      coreSegments.push(`C2=${inv.props.C2}`);
    }
  
    // Influencers
    // We'll generate something like: Influencers=[{LSMInfluencer,BuySign=true,...}|{...}]
    // Sort them in a stable, consistent manner. The Go code calls i.SortInfluencers().
    // Without knowing the exact sort criteria, we can do a simple alphabetical sort
    // by the influencer’s final DNA string.
    let inf = (inv.inf || []).slice(); // copy
    inf.sort((a, b) => influencerDNA(a).localeCompare(influencerDNA(b)));
    const infString = inf.map(obj => influencerDNA(obj)).join("|");
    coreSegments.push(`Influencers=[${infString}]`);
  
    // PDDInfluencers (if any)
    let pddInf = (inv.pddinf || []).slice(); // copy
    if (pddInf.length > 0) {
      // Sort them similarly
      pddInf.sort((a, b) => influencerDNA(a).localeCompare(influencerDNA(b)));
      const pddString = pddInf.map(obj => influencerDNA(obj)).join("|");
      coreSegments.push(`PDDInfluencers=[${pddString}]`);
    }
  
    // Strategy
    if (inv.props.Strategy) {
      coreSegments.push(`Strategy=${inv.props.Strategy}`);
    }
  
    // 2) Join them with semicolons => "C1=...;C2=...;Influencers=[...];PDDInfluencers=[...];Strategy=..."
    return coreSegments.join(";");
  }
  
  /**
   * Produces the DNA string for a single Influencer object, matching
   * the format from lsminfluencer.go:
   *
   * {LSMInfluencer,BuySign=...,Delta1=...,Delta2=...,Metric=...}
   *
   * If the Go code has more properties, you’d include them here in the same
   * alphabetical-after-subclass order. The snippet from `lsminfluencer.go`
   * shows a *fixed* order—thus we replicate that here.
   *
   * @param {object} infObj - The influencer object from `inv.inf[]` or `inv.pddinf[]`.
   * @returns {string} e.g. "{LSMInfluencer,BuySign=true,Delta1=-20,Delta2=-50,Metric=BuildingPermits}"
   */
  function influencerDNA(infObj) {
    // In Go, "LSMInfluencer" is always first, and the other fields come after,
    // usually in alphabetical order. The sample code is:
    //   return fmt.Sprintf("{%s,BuySign=%t,Delta1=%d,Delta2=%d,Metric=%s}", ...)
    //
    // So we replicate exactly that ordering in JS:
    let subclass = "LSMInfluencer";  // in the DNA, it is "LSMInfluencer" = ""
    // If for any reason infObj doesn't have LSMInfluencer, you could fallback,
    // but presumably they all do.
  
    let buySign = infObj.BuySign || "false"; // might be "true" or "false"
    let delta1 = infObj.Delta1 || "0";
    let delta2 = infObj.Delta2 || "0";
    let metric = infObj.Metric || "unknown";
  
    return `{${subclass},BuySign=${buySign},Delta1=${delta1},Delta2=${delta2},Metric=${metric}}`;
  }
  
  /**
   * Computes the SHA-256-based ID from the "core" DNA string, mimicking
   * the Go code’s HashDNA(dna string) -> hex output.
   *
   * @param {object} inv - The parsed investor object (from DNAParse).
   * @returns {string} A SHA-256 hex hash (64 hex characters).
   */
  function computeDNAID(inv) {
    // 1) Build the core string
    const coreString = buildDNACoreString(inv);
  
    // 2) Compute the SHA-256 hash of coreString
    const hashHex = sha256(coreString);  // using your existing sha256() function
  
    return hashHex;
  }
  
  /**
   * Verifies that the ID in inv.props.Options.ID matches the ID computed
   * from the core DNA string. Returns true if they match, false otherwise.
   *
   * @param {object} inv - The investor object from DNAParse.
   * @returns {boolean} True if the IDs match, otherwise false.
   */
  function verifyDNAID(inv) {
    // 1) Compute what the ID *should* be
    const computedID = computeDNAID(inv);
  
    // 2) Compare to the ID in the investor’s Options (if any)
    if (inv.props.Options && inv.props.Options.ID) {
      return (inv.props.Options.ID.toLowerCase() === computedID.toLowerCase());
    }
  
    // If no ID in Options, or something else is missing, return false
    return false;
  }