// List of common currencies and their symbols
const currencySymbols = {
    USD: 'United States Dollar ($)',
    JPY: 'Japanese Yen (¥)',
    EUR: 'Euro (€)',
    GBP: 'British Pound (£)',
    CAD: 'Canadian Dollar (C$)',
    AUD: 'Australian Dollar (A$)',
    CHF: 'Swiss Franc (Fr)',
    CNY: 'Chinese Yuan (¥)',
    INR: 'Indian Rupee (₹)',
    KRW: 'South Korean Won (₩)',
    // Add more currencies as needed
};

class DNARow {
    constructor(tableID, actionCallback = null, actionConfig = {}) {
        this.table = document.getElementById(tableID);
        this.actionCallback = actionCallback;
        this.dnaStrings = []; // Store DNA strings to map indices
        this.weightDisplay = new WeightDisplay('compact');

        // Default action configuration
        this.actionConfig = {
            showCopy: true,
            showEdit: true,
            showDelete: true,
            ...actionConfig  // Override defaults with provided config
        };
    }

    /**
     * Updates the weight display style. If there's existing data, it will be re-rendered with the new style.
     * @param {string} style - The style to use for weight display. Currently
     *                         supported styles are: classic, greek, modern, compact.
     */
    setWeightStyle(style) {
        this.weightDisplay = new WeightDisplay(style);
        // Refresh the display if there's existing data
        if (this.dnaStrings.length > 0) {
            this.DNARowRender(this.dnaStrings);
        }
    }

    /**
     * Sets or updates the callback function for DNA row actions
     * @param {Function} callback - Function to call when actions occur, receives (action, index)
     */
    setActionCallback(callback) {
        this.actionCallback = callback;
    }

    /**
     * DNARowRender - Renders DNA strings into the table
     * @param {Array<string>} dnaStrings - An array of DNA strings to parse and render
     */
    DNARowRender(dnaStrings) {
        this.dnaStrings = [...dnaStrings]; // Store reference to maintain indices
        const parsedDNA = dnaStrings.map(dna => DNAParse(dna)).filter(inv => inv !== null);
        const dnaHTMLArray = parsedDNA.map((inv, index) => this.generateDNAHTML(inv, index));
        this.insertDNARows(dnaHTMLArray);
        this.bindDNARowEvents();
    }

    /**
     * Binds event listeners for DNA row actions and influencer sections.
     * 
     * This function adds click handlers to DNA row action buttons to trigger
     * the specified action callback with the action type and row index. It 
     * also adds handlers for influencer sections to toggle dropdown visibility 
     * and a global event listener to close any open popups when clicking outside.
     */
    bindDNARowEvents() {
        // Add click handlers for action buttons
        this.table.querySelectorAll('.dnarow-icon-button').forEach(button => {
            button.addEventListener('click', (event) => {
                event.stopPropagation();

                if (this.actionCallback) {
                    const action = button.dataset.action;
                    const index = parseInt(button.dataset.index, 10);

                    // Make sure we get the button element even if SVG was clicked
                    const targetButton = event.target.closest('.dnarow-icon-button');

                    this.actionCallback(action, index).then(() => {
                        // Show UI feedback
                        switch (action) {
                            case 'copy':
                                targetButton.classList.add('copied');
                                setTimeout(() => targetButton.classList.remove('copied'), 2000);
                                break;
                            case 'edit':
                                break;
                            case 'delete':
                                break;
                        }
                    });
                }
            });
        });

        // Add click handlers for influencer sections
        document.querySelectorAll('.dnarow-influencer-section').forEach(section => {
            section.addEventListener('click', function (event) {
                event.stopPropagation();
                const dropdown = this.querySelector('.dnarow-dropdown-content');
                const allPopups = document.querySelectorAll('.dnarow-popup, .dnarow-dropdown-content');

                // Close all other popups and dropdowns
                allPopups.forEach(el => {
                    if (el !== dropdown) {
                        el.style.display = 'none';
                    }
                });

                // Toggle on this dropdown
                dropdown.style.display = dropdown.style.display === 'block' ? 'none' : 'block';
            });
        });

        // Close all popups when clicking outside
        document.addEventListener('click', function () {
            const allPopups = document.querySelectorAll('.dnarow-popup, .dnarow-dropdown-content');
            allPopups.forEach(el => el.style.display = 'none');
        });
    }


    /**
     * Escapes HTML special characters in a string to prevent injection.
     *
     * @param {string} str - The string to escape.
     * @returns {string} The escaped string.
     */
    escapeHTML(str) {
        if (typeof str !== 'string') {
            return str;
        }
        return str.replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    /**
     * Generates the HTML string for a single DNA row.
     *
     * @param {object} inv - The investor object returned by DNAParse.
     * @returns {string} The HTML string representing the dna-row.
     */
    generateDNAHTML(inv, index) {
        // Start building the dna-row div
        let html = `<div class="dna-row">`;

        // Checkbox Input
        html += `<input type="checkbox" data-index="${index}">`;

        //----------------------------
        // ID Section
        //----------------------------
        if (inv.props.ID) {
            const fullID = this.escapeHTML(inv.props.ID);
            const shortID = `${fullID.slice(0, 5)}&hellip;${fullID.slice(-3)}`;
            html += `
                    <span class="dnarow-id-section" onclick="DNARow.togglePopup(event, 'id')">
                        ID: ${shortID}
                        <div class="dnarow-popup">
                            <div class="dnarow-popup-header">
                                <span class="dnarow-popup-title">Full ID</span>
                                <span class="dnarow-close-popup" onclick="DNARow.closePopup(event)">×</span>
                            </div>
                            <div style="font-family: monospace; word-break: break-all;">
                                ${fullID}
                            </div>
                            <button class="dnarow-copy-button" onclick="DNARow.copyId(event)"> <span class="dnarow-copy-icon">&#x2398;</span> <span class="copy-text">Copy ID</span> </button>
                        </div>
                    </span>
                `;
        }

        //----------------------------
        // Name Button
        //----------------------------
        let fullName = ' ';
        if (inv.props.Options.FirstName && inv.props.Options.LastName) {
            fullName = `${inv.props.Options.FirstName} ${inv.props.Options.LastName}`;
        } else if (inv.props.Name) {
            fullName = inv.props.Name;
        }

        html += `
                <span class="dnarow-name-button" onclick="DNARow.togglePopup(event, 'name')">
                    <span class="dnarow-name-text">${this.escapeHTML(fullName)}</span>
                    <div class="dnarow-popup">
                        <div class="dnarow-popup-header">
                            <span class="dnarow-popup-title">Full Name</span>
                            <span class="dnarow-close-popup" onclick="DNARow.closePopup(event)">×</span>
                        </div>
                        <div class="dnarow-dnarow-full-name">${this.escapeHTML(fullName)}</div>
                    </div>
                </span>
            `;

        //----------------------------
        // Currency Pair
        //----------------------------
        let currencyPair = '';
        let currencyPopupContent = '';
        const C1 = inv.props.C1 || '';
        const C2 = inv.props.C2 || '';

        // Get currency descriptions or fallback to the code itself
        const C1Description = currencySymbols[C1] || C1;
        const C2Description = currencySymbols[C2] || C2;

        // Always populate the currencyPair and dnarow-popup content
        if (C1.length > 0 || C2.length > 0) {
            currencyPair = `${this.escapeHTML(C1)}/${this.escapeHTML(C2)}`;
        }
        currencyPopupContent = `
                <div><b>C1:</b> ${this.escapeHTML(C1Description)}</div>
                <div><b>C2:</b> ${this.escapeHTML(C2Description)}</div>
                <div><b>Strategy:</b> ${this.escapeHTML(inv.props.Strategy || 'Unknown')}</div>
            `;

        // Add to HTML
        const strategyClass =
            inv.props.Strategy === 'MajorityRules'
                ? 'dnarow-majority-rules'
                : inv.props.Strategy === 'DistributedDecision'
                    ? 'dnarow-distributed-decision'
                    : 'unknown-strategy'; // Default class if no strategy found

        html += `
                <span class="dnarow-currency-pair ${strategyClass}" onclick="DNARow.togglePopup(event, 'currency')">
                    <span class="dnarow-currency-text">${currencyPair}</span>
                    <div class="dnarow-popup">
                        <div class="dnarow-popup-header">
                            <span class="dnarow-popup-title">Currency Details</span>
                            <span class="dnarow-close-popup" onclick="DNARow.closePopup(event)">×</span>
                        </div>
                        ${currencyPopupContent}
                    </div>
                </span>`;

        //----------------------------
        // weights
        //----------------------------
        let weights = '';
        if (inv.BirthCertificate.Wprf) weights += `P:${this.escapeHTML(inv.BirthCertificate.Wprf)} `;
        if (inv.BirthCertificate.Wcor) weights += `R:${this.escapeHTML(inv.BirthCertificate.Wcor)} `;
        if (inv.BirthCertificate.Wcon) weights += `N:${this.escapeHTML(inv.BirthCertificate.Wcon)}`;
        if (weights.trim()) {
            html += this.weightDisplay.render(inv.BirthCertificate.Wprf, inv.BirthCertificate.Wcor, inv.BirthCertificate.Wcon, inv.BirthCertificate.DtStart, inv.BirthCertificate.DtStop);
        }

        //----------------------------
        // Influencers Section
        //----------------------------
        if (inv.inf && inv.inf.length > 0) {
            const count = inv.inf.length;
            html += `<span class="dnarow-influencer-section">
                    Influencers <span class="dnarow-count-badge">${count}</span>
                    <div class="dnarow-dropdown-content">`;
            inv.inf.forEach(inf => {
                const signalClass = inf.BuySign === 'true' ? 'dnarow-buy-signal' : 'dnarow-sell-signal';
                const signalSymbol = inf.BuySign === 'true' ? '+' : '−';
                const metric = this.escapeHTML(inf.Metric);
                const delta1 = this.escapeHTML(inf.Delta1);
                const delta2 = this.escapeHTML(inf.Delta2);
                html += `<div class="dnarow-metric-item">
                        <span class="${signalClass}">${signalSymbol}</span> 
                        ${metric} (${delta1},${delta2})
                    </div>`;
            });
            html += `</div></span>`;
        }

        //----------------------------
        // PDDInfluencers Section
        //----------------------------
        if (inv.pddinf && inv.pddinf.length > 0) {
            const count = inv.pddinf.length;
            html += `<span class="dnarow-influencer-section">
                        <span class="dnarow-pdd-badge">PDD</span>Influencers <span class="dnarow-count-badge">${count}</span>
                        <div class="dnarow-dropdown-content">`;
            inv.pddinf.forEach(inf => {
                const signalClass = inf.BuySign === 'true' ? 'dnarow-buy-signal' : 'dnarow-sell-signal';
                const signalSymbol = inf.BuySign === 'true' ? '+' : '−';
                const metric = this.escapeHTML(inf.Metric);
                const delta1 = this.escapeHTML(inf.Delta1);
                const delta2 = this.escapeHTML(inf.Delta2);
                html += `<div class="dnarow-metric-item">
                            <span class="${signalClass}">${signalSymbol}</span> 
                            ${metric} (${delta1},${delta2})
                        </div>`;
            });
            html += `</div></span>`;
        }

        // Replace the existing action icons section with this:
        if (Object.values(this.actionConfig).some(v => v)) {  // Only add actions div if any actions are enabled
            html += '<div class="dnarow-actions">';

            if (this.actionConfig.showCopy) {
                html += `
            <button class="dnarow-icon-button" title="Copy DNA" data-action="copy" data-index="${index}">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="9" y="9" width="13" height="13" rx="2" ry="2"></rect>
                    <path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"></path>
                </svg>
            </button>`;
            }

            if (this.actionConfig.showEdit) {
                html += `
            <button class="dnarow-icon-button" title="Edit DNA" data-action="edit" data-index="${index}">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M17 3a2.828 2.828 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5L17 3z"></path>
                </svg>
            </button>`;
            }

            if (this.actionConfig.showDelete) {
                html += `
            <button class="dnarow-icon-button delete-button" title="Delete DNA" data-action="delete" data-index="${index}">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M3 6h18"></path>
                    <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6"></path>
                    <path d="M8 6V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                </svg>
            </button>`;
            }

            html += '</div>';
        }

        // Close dna-row div
        html += `</div>`;

        return html;
    }

    /**
     * Inserts generated DNA rows into the table.  This implementation assumes that
     * the caller manages any list it wants to keep. The caller can always use
     * the `DeleteAllDNARows` method to clear the table before inserting new rows.
     *
     * @param {Array<string>} dnaHTMLArray - An array of HTML strings for each dna-row.
     */
    insertDNARows(dnaHTMLArray) {
        dnaHTMLArray.forEach(html => {
            const row = this.table.insertRow();
            const cell = row.insertCell(0);
            cell.innerHTML = html;
        });
    }

    /**
     * Deletes all DNA rows from the table except for the header.
     *
     * Iterates over the rows in the table, starting from the second row (index 1),
     * and deletes each row until only the header remains.
     */
    DeleteAllDNARows() {
        if (this.table == null) {
            return;
        }
        // Get all rows in the table
        const rows = this.table.rows;

        // Loop through rows, starting from the second row (index 1)
        while (rows.length > 1) {
            this.table.deleteRow(1); // Always delete the row at index 1
        }
    }

    /**
     * Handles toggling of popups and dropdowns.
     *
     * @param {Event} event - The click event.
     * @param {string} type - The type of popup ('id' or 'name').
     */
    static togglePopup(event, type) {
        event.stopPropagation();
        const popup = event.currentTarget.querySelector('.dnarow-popup');
        const allPopups = document.querySelectorAll('.dnarow-popup, .dnarow-dropdown-content');

        // Close all other popups and dropdowns
        allPopups.forEach(el => {
            if (el !== popup) {
                el.style.display = 'none';
            }
        });

        // Toggle this popup
        popup.style.display = popup.style.display === 'block' ? 'none' : 'block';
    }

    /**
     * Closes the dnarow-popup when the close button is clicked.
     *
     * @param {Event} event - The click event.
     */
    static closePopup(event) {
        event.stopPropagation();
        const popup = event.target.closest('.dnarow-popup, .dnarow-dropdown-content');
        if (popup) {
            popup.style.display = 'none';
        }
    }

    /**
     * Copies the full ID to the clipboard.
     *
     * @param {Event} event - The click event.
     */
    static copyId(event) {
        event.stopPropagation();
        // Extract the ID from the parent .dnarow-id-section
        const idSection = event.target.closest('.dnarow-id-section');
        const fullID = idSection.querySelector('.dnarow-popup div[style]').textContent;
        navigator.clipboard.writeText(fullID).then(() => {
            const button = event.target;
            const originalText = button.textContent;
            button.textContent = '✓ Copied!';
            setTimeout(() => {
                button.textContent = originalText;
            }, 2000);
        }).catch(err => {
            console.error('Failed to copy ID: ', err);
        });
    }

    /**
     * Returns an array of indices corresponding to checked rows in the table.
     * @returns {Array<number>} Array of indices for checked rows, empty if none are checked.
     */
    getCheckedIndices() {
        const checked = [];
        this.table.querySelectorAll('.dna-row input[type="checkbox"]').forEach((checkbox, index) => {
            if (checkbox.checked) {
                checked.push(index);
            }
        });
        return checked;
    }

    /**
     * Sets the checked state of checkboxes based on provided indices.
     * Unchecks all checkboxes first, then checks only those whose indices
     * are in the provided array and exist in the table.
     * 
     * @param {Array<number>} indices - Array of indices to be checked
     */
    setCheckedIndices(indices) {
        // Get all checkboxes in the DNA rows
        const checkboxes = this.table.querySelectorAll('.dna-row input[type="checkbox"]');

        // First uncheck all checkboxes
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
        });

        // Now check only the valid indices
        indices.forEach(index => {
            if (index >= 0 && index < checkboxes.length) {
                checkboxes[index].checked = true;
            }
        });
    }
}