class WeightDisplay {
    constructor(style = 'compact') {
        this.style = style;
        
        // Clean up any existing styles and orphaned elements
        const existingStyles = document.querySelectorAll('style[data-weight-display]');
        existingStyles.forEach(style => style.remove());
        
        const orphanedPopups = document.querySelectorAll('.dnarow-weights-popup');
        orphanedPopups.forEach(popup => popup.remove());
        
        this.addStyles();
    }

    addStyles() {
        const styleElement = document.createElement('style');
        styleElement.setAttribute('data-weight-display', 'true');
        styleElement.textContent = `
            /* Base weight styles */
            .dnarow-weight-letter {
                font-family: serif;
                font-weight: bold;
                color: #b4483e;
            }
            
            .dnarow-weight-number {
                font-family: monospace;
                font-size: 0.875em;
                margin-left: 1px;
            }
            
            /* Common background styles for all variants */
            .dnarow-weights-classic,
            .dnarow-weights-greek,
            .dnarow-weights-modern .dnarow-weight-pair,
            .dnarow-weights-compact {
                position: relative;
                background: #FDF1F0;
                border: 1px solid #E15A4E;
                border-radius: 4px;
                padding: 2px 6px;
                cursor: pointer;
            }
            
            /* Compact style (default) */
            .dnarow-weights-compact {
                display: inline-flex;
                align-items: center;
            }
            
            .dnarow-weights-compact .dnarow-weight-separator {
                color: #E15A4E;
                margin: 0 0.5rem;
                opacity: 0.7;
            }

            .dnarow-weight-pair {
                display: inline-flex;
                align-items: baseline;
            }

            /* Hover effect */
            .dnarow-weights-compact:hover,
            .dnarow-weights-classic:hover,
            .dnarow-weights-greek:hover,
            .dnarow-weights-modern .dnarow-weight-pair:hover {
                background: #FDE8E6;
                border-color: #D14D41;
            }

            /* Popup styles */
            .dnarow-weights-popup {
                display: none;
                position: absolute;
                top: calc(100% + 4px);
                left: 0;
                background: white;
                border: 1px solid #ccc;
                border-radius: 4px;
                padding: 8px;
                box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
                z-index: 1000;
                min-width: 250px;
            }

            .dnarow-weights-popup table {
                width: 100%;
                border-collapse: collapse;
                margin: 0;
            }

            .dnarow-weights-popup th {
                text-align: center;
                padding: 4px 8px;
                border-bottom: 1px solid #eee;
                color: #666;
            }

            .dnarow-weights-popup td {
                padding: 4px 8px;
                border-bottom: 1px solid #eee;
            }

            .dnarow-weights-popup td:last-child {
                text-align: right;
                font-family: monospace;
            }

            .dnarow-weights-popup .close-popup {
                position: absolute;
                top: 8px;
                right: 8px;
                cursor: pointer;
                color: #666;
                font-size: 18px;
            }

            .dnarow-weights-popup .weight-letter {
                font-family: serif;
                font-weight: bold;
                color: #b4483e;
            }
        `;
        document.head.appendChild(styleElement);
    }

    formatNumber(value) {
        const numStr = typeof value === 'string' ? 
            parseFloat(value).toFixed(4) : 
            value.toFixed(4);
        return '.' + numStr.split('.')[1];
    }

    formatPercent(value) {
        return (value * 100).toFixed(2) + '%';
    }

    renderPopupContent(wprf, wcor, wcon,dtStart,dtStop) {
        // Only render if we have actual values
        if (wprf === undefined || wcor === undefined || wcon === undefined) {
            return '';
        }

        const fragment = document.createElement('div');
        fragment.className = 'dnarow-weights-popup';
        fragment.innerHTML = `
            <span class="close-popup" onclick="WeightDisplay.closePopup(event)">×</span>
            <table>
                <tr>
                    <th colspan="2">Fitness Score Weighting</th>
                </tr>
                <tr>
                    <td>Profitability (<span class="weight-letter">P</span>):</td>
                    <td>${this.formatPercent(wprf)}</td>
                </tr>
                <tr>
                    <td>Correctness (<span class="weight-letter">R</span>):</td>
                    <td>${this.formatPercent(wcor)}</td>
                </tr>
                <tr>
                    <td>Consistency (<span class="weight-letter">N</span>):</td>
                    <td>${this.formatPercent(wcon)}</td>
                </tr>
                <tr>
                    <td colspan="2">
                    Birth Period: ${dtStart} - ${dtStop}
                    </td>
                </tr>
            </table>`;
        return fragment.outerHTML;
    }

    renderPair(letter, value) {
        if (value === undefined) return '';
        return `
            <span class="dnarow-weight-pair">
                <span class="dnarow-weight-letter">${letter}</span>
                <span class="dnarow-weight-number">${this.formatNumber(value)}</span>
            </span>
        `;
    }

    render(wprf, wcor, wcon, dtStart, dtStop) {
        // Only render if we have actual values
        if (wprf === undefined || wcor === undefined || wcon === undefined) {
            return '';
        }

        const weights = [
            { letter: 'P', value: wprf },
            { letter: 'R', value: wcor },
            { letter: 'N', value: wcon }
        ];

        const container = document.createElement('div');
        container.className = 'dnarow-weights dnarow-weights-compact';
        container.setAttribute('onclick', 'WeightDisplay.togglePopup(event)');
        
        container.innerHTML = `
            ${weights.map((w, i) => `
                ${this.renderPair(w.letter, w.value)}
                ${i < weights.length - 1 ? 
                    '<span class="dnarow-weight-separator">|</span>' : 
                    ''}
            `).join('')}
            ${this.renderPopupContent(wprf, wcor, wcon, dtStart, dtStop)}`;
        
        return container.outerHTML;
    }

    static togglePopup(event) {
        event.stopPropagation();
        const currentElement = event.currentTarget;
        const popup = currentElement.querySelector('.dnarow-weights-popup');
        const allPopups = document.querySelectorAll('.dnarow-popup, .dnarow-dropdown-content, .dnarow-weights-popup');

        allPopups.forEach(el => {
            if (el !== popup) {
                el.style.display = 'none';
            }
        });

        popup.style.display = popup.style.display === 'block' ? 'none' : 'block';

        if (popup.style.display === 'block') {
            const rect = popup.getBoundingClientRect();
            const viewportWidth = window.innerWidth;
            const viewportHeight = window.innerHeight;

            if (rect.right > viewportWidth) {
                popup.style.left = 'auto';
                popup.style.right = '0';
            }

            if (rect.bottom > viewportHeight) {
                popup.style.top = 'auto';
                popup.style.bottom = 'calc(100% + 4px)';
            }
        }
    }

    static closePopup(event) {
        event.stopPropagation();
        const popup = event.target.closest('.dnarow-weights-popup');
        if (popup) {
            popup.style.display = 'none';
        }
    }
}

// Add global click handler to close popups when clicking outside
document.addEventListener('click', function() {
    const allPopups = document.querySelectorAll('.dnarow-popup, .dnarow-dropdown-content, .dnarow-weights-popup');
    allPopups.forEach(el => el.style.display = 'none');
});