#!/bin/bash

LOG_FILE="./macplatoinstall.log"
PLATO_INSTALL_DIR="/usr/local/plato"
SIMD_DATA_DIR="/var/lib/simd"
VERBOSE=0
VERSIONCHECKONLY=0
UNINSTALL=0
ROLLBACK=0

#---------------------------------------------------------------
# Function to echo messages in verbose mode
#---------------------------------------------------------------
vecho() {
    if [ $VERBOSE -eq 1 ]; then
        echo "$@"
    fi
}

#---------------------------------------------------------------
# doInstall - the main function, install plato
#---------------------------------------------------------------
doInstall() {
    #------------------------------------------------
    # Make sure we are in the correct directory
    #------------------------------------------------
    if [ ! -f "./bin/simulator" ]; then
        DIR=$(dirname "$0")
        vecho "Changing directory to $DIR..."
        cd "$DIR" || exit 1
        if [ ! -f "./bin/simulator" ]; then
            echo "This script must be run from the directory containing macplatoinstall.sh."
            exit 1
        fi
    fi

    #------------------------------------------------
    # Stop any running instances of simulator and simtalk
    #------------------------------------------------
    vecho "Checking for running instances of simulator..."
    simulators=$(pgrep simulator)
    if [ -n "$simulators" ]; then
        COUNT=$(echo "$simulators" | wc -l)
        vecho "Found $COUNT instances of simulator. Terminating..."
        killall simulator
    else
        vecho "No running instances of simulator found."
    fi

    vecho "Checking for running instances of simtalk..."
    simtalks=$(pgrep simtalk)
    if [ -n "$simtalks" ]; then
        vecho "Found simtalk. Shutting down simtalk..."
        curl -s http://localhost:8251/Shutdown
        sleep 1
        echo
        simtalks=$(pgrep simtalk)
        if [ -n "$simtalks" ]; then
            vecho "Killing all remaining simtalk instances..."
            killall simtalk
        fi
    else
        vecho "No running instances of simtalk found."
    fi

    #------------------------------------------------
    # Ensure that we have the simd user and group...
    #------------------------------------------------
    if id "simd" &>/dev/null; then
        vecho "Skipping user creation, user 'simd' already exists."
    else
        vecho "Creating user 'simd'..."
        next_uid=$(dscl . -list /Users UniqueID | awk '{uid[$2]=1} END {for (i=501; i<600; i++) if (!uid[i]) {print i; exit}}')
        dscl . -create /Users/simd
        dscl . -create /Users/simd UserShell /usr/bin/false
        dscl . -create /Users/simd RealName "SIMD Service User"
        dscl . -create /Users/simd UniqueID "$next_uid"
        dscl . -create /Users/simd PrimaryGroupID "$next_uid"
        dscl . -create /Users/simd NFSHomeDirectory /var/empty
        dscl . -passwd /Users/simd "Foolme123"
        dscl . -append /Groups/wheel GroupMembership simd
    fi

    #---------------------------------------------------------------
    # Ensure the group 'simd' exists and set the correct GID
    #---------------------------------------------------------------
    if [ "$EUID" -eq 0 ]; then
        if ! dscl . -list /Groups PrimaryGroupID | grep -q "simd"; then
            vecho "Creating group 'simd' with GID $next_gid..."
            next_gid=$(dscl . -list /Groups PrimaryGroupID | awk '{gid[$2]=1} END {for (i=1000; i<60000; i++) if (!gid[i]) {print i; exit}}')
            dscl . -create /Groups/simd
            dscl . -create /Groups/simd PrimaryGroupID "$next_gid"
            dscl . -append /Groups/simd GroupMembership simd
        else
            vecho "Skipping group creation, group 'simd' already exists."
            next_gid=$(dscl . -read /Groups/simd PrimaryGroupID | awk '{print $2}')
        fi

        #---------------------------------------------------------------
        # Update PrimaryGroupID for user 'simd' to match group 'simd'
        #---------------------------------------------------------------
        vecho "Updating PrimaryGroupID for user 'simd' to match group 'simd'..."
        dscl . -create /Users/simd PrimaryGroupID "$next_gid"

    fi

    #------------------------------------------------
    # Ensure /usr/local/plato exists
    #------------------------------------------------
    if [ ! -d "$PLATO_INSTALL_DIR" ]; then
        vecho "Creating $PLATO_INSTALL_DIR..."
        mkdir -p "$PLATO_INSTALL_DIR"
        if [ "$EUID" -eq 0 ]; then
            chown -R simd:simd "$PLATO_INSTALL_DIR"
        fi
    fi

    #------------------------------------------------
    # Backup the existing release...
    #------------------------------------------------
    vecho "Backing up current release to ${SIMD_DATA_DIR}/platobkup/"
    if [ -d "${PLATO_INSTALL_DIR}/bin" ]; then
        vecho "Backing up ${PLATO_INSTALL_DIR}/bin..."
        mkdir -p "${SIMD_DATA_DIR}/platobkup"
        rm -rf "${SIMD_DATA_DIR}/platobkup/"*
        mv "${PLATO_INSTALL_DIR}"/* "${SIMD_DATA_DIR}/platobkup/"
    fi

    #------------------------------------------------
    # Copy the new release to the release location
    #------------------------------------------------
    vecho "Copying new release to ${PLATO_INSTALL_DIR}..."
    cp -r ./bin ./etc ./html ./man ${PLATO_INSTALL_DIR}/

    #---------------------------------------------------------------
    # Set the owner of the simq released files to the simd user
    #---------------------------------------------------------------
    vecho "Setting ownership of ${PLATO_INSTALL_DIR} to 'simd'..."
    if [ "$EUID" -eq 0 ]; then
        chown -R simd:simd "$PLATO_INSTALL_DIR"
    fi

    #------------------------------------------------------------
    # Copy any critical config files from the backup release...
    #------------------------------------------------------------
    if [ -f /var/lib/simd/platobkup/bin/extres.json5 ]; then
        cp /var/lib/simd/platobkup/bin/extres.json5 "${PLATO_INSTALL_DIR}/bin/"
        chown simd:simd "${PLATO_INSTALL_DIR}/bin/extres.json5"
    else
        cat >"${PLATO_INSTALL_DIR}/bin/extres.json5" <<EOF
{
    "TradingeconomicsAPIKey": "placeholder",
    "DbUser": "placeholder",
    "DbPass": "placeholder",
}
EOF
    fi

    #---------------------------------------------------------------
    # Validate the installed files by running simulator and simtalk
    # to get their version strings
    #---------------------------------------------------------------
    vecho "Checking newly installed versions..."
    validate_plato_version "${PLATO_INSTALL_DIR}/bin/simulator" "Simulator"
    validate_plato_version "${PLATO_INSTALL_DIR}/bin/simtalk" "Simtalk"

}

#-----------------------------------------------------------------------------
# Validates the version string of a given executable.
#
# Parameters:
#   executable_path (string): The path to the executable.
#   expected_product (string): The expected product name.
#
# Returns:
#   0 if the version string is valid, 1 otherwise.
#-----------------------------------------------------------------------------
validate_plato_version() {
    local executable_path="$1"
    local expected_product="$2"

    # Check if executable exists and is executable
    if [[ ! -x "$executable_path" ]]; then
        echo "Error: $executable_path is not executable or does not exist"
        return 1
    fi

    # Execute the program with -v option and capture the output
    local version_string=$("$executable_path" -v)
    if [[ $? -ne 0 ]]; then
        echo "Error: Failed to execute $executable_path -v"
        return 1
    fi

    # Check overall format and extract components
    if [[ $version_string =~ ^PLATO\ ([A-Za-z]+)\ version\ ([0-9]+)\.([0-9]+)-([0-9]{4})([0-9]{2})([0-9]{2})T([0-9]{2})([0-9]{2})([0-9]{2})$ ]]; then
        local product_name="${BASH_REMATCH[1]}"
        local major="${BASH_REMATCH[2]}"
        local minor="${BASH_REMATCH[3]}"
        local year="${BASH_REMATCH[4]}"
        local month="${BASH_REMATCH[5]}"
        local day="${BASH_REMATCH[6]}"
        local hour="${BASH_REMATCH[7]}"
        local minute="${BASH_REMATCH[8]}"
        local second="${BASH_REMATCH[9]}"
    else
        echo "version string format looks invalid"
        echo "Received: $version_string"
        echo "Expected format: PLATO <ProductName> version <Major>.<Minor>-<YYYY><MM><DD>T<HH><MM><SS>"
        return 1
    fi

    # Validate product name
    if [[ "$product_name" != "$expected_product" ]]; then
        echo "version string Product Name looks invalid"
        echo "Expected: $expected_product"
        echo "Received: $product_name"
        return 1
    fi

    # Validate major and minor version numbers
    if ! [[ $major =~ ^[1-9][0-9]*$ ]]; then
        echo "version string Major version number looks invalid"
        echo "Received: $major"
        echo "Expected: A positive integer"
        return 1
    fi

    if ! [[ $minor =~ ^[0-9]+$ ]]; then
        echo "version string Minor version number looks invalid"
        echo "Received: $minor"
        echo "Expected: A non-negative integer"
        return 1
    fi

    # Validate date components
    if ! [[ $month =~ ^(0[1-9]|1[0-2])$ ]]; then
        echo "version string Date looks invalid"
        echo "Invalid month: $month"
        echo "Expected: 01-12"
        return 1
    fi

    if ! [[ $day =~ ^(0[1-9]|[12][0-9]|3[01])$ ]]; then
        echo "version string Date looks invalid"
        echo "Invalid day: $day"
        echo "Expected: 01-31"
        return 1
    fi

    # Validate time components
    if ! [[ $hour =~ ^([01][0-9]|2[0-3])$ ]]; then
        echo "version string Time looks invalid"
        echo "Invalid hour: $hour"
        echo "Expected: 00-23"
        return 1
    fi

    if ! [[ $minute =~ ^[0-5][0-9]$ ]]; then
        echo "version string Time looks invalid"
        echo "Invalid minute: $minute"
        echo "Expected: 00-59"
        return 1
    fi

    if ! [[ $second =~ ^[0-5][0-9]$ ]]; then
        echo "version string Time looks invalid"
        echo "Invalid second: $second"
        echo "Expected: 00-59"
        return 1
    fi

    echo "New version string looks good"
    return 0
}

#------------------------------------------------------------------------------------
# Reverts the installation to its previous state by restoring the backed-up files.
#------------------------------------------------------------------------------------
rollback() {
    vecho "Rolling back..."
    if [ -d "${SIMD_DATA_DIR}/platobkup" ]; then
        vecho "Restoring ${PLATO_INSTALL_DIR}/..."
        mkdir -p "${PLATO_INSTALL_DIR:?}"
        rm -rf "${PLATO_INSTALL_DIR:?}/bin"
        rm -rf "${PLATO_INSTALL_DIR:?}/etc"
        rm -rf "${PLATO_INSTALL_DIR:?}/html"
        rm -rf "${PLATO_INSTALL_DIR:?}/man"
        cp -r "${SIMD_DATA_DIR}/platobkup/"* "${PLATO_INSTALL_DIR}/"
        if [ "$EUID" -eq 0 ]; then
            chown -R simd:simd "${PLATO_INSTALL_DIR}"
        fi
    fi
}

# Uninstalls the Plato installation by removing the PLATO_INSTALL_DIR directory.
uninstall() {
    vecho "Uninstalling..."
    if [ -d "${PLATO_INSTALL_DIR}" ]; then
        vecho "Removing ${PLATO_INSTALL_DIR}..."
        rm -rf "${PLATO_INSTALL_DIR:?}"
        rm -rf "${PLATO_INSTALL_DIR:?}/man"
    fi
}

#------------------------------------------------
# Process command-line options
#------------------------------------------------
while getopts "vRUV" opt; do
    case $opt in
    v)
        VERBOSE=1
        ;;
    V)
        VERBOSE=1
        VERSIONCHECKONLY=1
        ;;
    R)
        ROLLBACK=1
        ;;
    U)
        UNINSTALL=1
        ;;
    *)
        echo "Usage: $0 [-v]" >&2
        exit 1
        ;;
    esac
done

ID=$(whoami)
if [ "${ID}" != "root" ] && [ "${ID}" != "simd" ]; then
    echo "This script must be run as root."
    exit 1
fi

#--------------------------------------------------------------------------------------------
# Ensure the script is running as root. It needs to be root because there are commands that
# create users and groups.
#--------------------------------------------------------------------------------------------
if id "simd" &>/dev/null; then
    echo "simd user already exists. Proceeding without root requirement."
elif [ "$EUID" -ne 0 ] && [ "$VERSIONCHECKONLY" -ne 1 ]; then
    echo "This script must be run as root. Please use or run as root user."
    exit 1
fi

#------------------------------------------------
# Redirect all output to the log file
#------------------------------------------------
exec > >(tee -a "$LOG_FILE") 2>&1
vecho "-----------------------------------------------------------------------------"
TIME=$(date +"%Y-%m-%d %H:%M:%S")
vecho "Start time: ${TIME}"

if ((UNINSTALL == 1)); then
    uninstall
elif ((ROLLBACK == 1)); then
    rollback
elif ((VERSIONCHECKONLY == 1)); then
    validate_plato_version "${PLATO_INSTALL_DIR}/bin/simulator" "Simulator"
    validate_plato_version "${PLATO_INSTALL_DIR}/bin/simtalk" "Simtalk"
else
    doInstall
fi

echo "Installation complete!"
if [ "$EUID" -ne 0 ]; then
    if [ -f ./"${LOG_FILE}" ]; then
        mv ./"${LOG_FILE}" ../
    fi
fi
