// msgbar.js

/**
 * Initialize the message bar by adding a click event listener to the close button.
 *
 * @since 0.1.0
 */
function initMsgbar() {
    // Use event delegation for the close button
    $("#messageBar").on("click", ".close-button", function() {
        hideMessageBar();
    });
}

/**
 * Show a message bar with a message and an icon based on the given type.
 * Supported types are: "error", "warning", "success", and "notification".
 *
 * @param {string} type The type of the message.
 * @param {string} message The message to display.
 *
 * @since 0.1.0
 */
function showMessageBar(type, message) {
    MessageHistory.addMessage(type, message);
}

function hideMessageBar() {
    const $messageBar = $("#messageBar");

    if ($messageBar.length === 0) {
        console.error("messageBar not found!");
        return;
    }

    // Hide the messageBar by removing 'visible' and adding 'hidden'
    $messageBar.removeClass("visible").addClass("hidden");
}

const MessageHistory = {
    messages: [],
    maxHistory: 25,  // Keep last 25 messages
    visibleCount: 3, // Show last 3 messages by default

    addMessage(type, message) {
        const timestamp = new Date();
        this.messages.push({ type, message, timestamp });
        
        // Keep only the last maxHistory messages
        if (this.messages.length > this.maxHistory) {
            this.messages = this.messages.slice(-this.maxHistory);
        }
        
        this.updateDisplay();
    },

    updateDisplay() {
        const $messageBar = $("#messageBar");
        const $messageContent = $("#messageContent");
        
        // Get the last n messages, newest first
        const visibleMessages = this.messages.slice(-this.visibleCount).reverse();
        
        if (visibleMessages.length === 0) {
            $messageBar.removeClass("visible").addClass("hidden");
            return;
        }
    
        // Create message elements
        const messageHTML = `
            <div class="messages-container">
                ${visibleMessages.map(msg => `
                    <div class="message-line ${msg.type}">
                        <span class="message-icon">${this.getIcon(msg.type)}</span>
                        <span class="message-content">${msg.message}</span>
                    </div>
                `).join('')}
            </div>
            <button class="close-button">×</button>
        `;
    
        // Update the message bar content
        $messageBar.find("> div").html(messageHTML);
        
        // Show the message bar
        $messageBar.removeClass("hidden").addClass("visible");
    },

    getIcon(type) {
        switch (type) {
            case "error": return "❌";
            case "warning": return "⚠️";
            case "success": return "✅";
            case "notification": return "ℹ️";
            default: return "ℹ️";
        }
    }
};

window.initMsgbar = initMsgbar;
window.showMessageBar = showMessageBar;
window.hideMessageBar = hideMessageBar;
window.MessageHistory = MessageHistory;