import appState from './state.js';

class ViewManager {
    constructor() {
        this.baseUrl = '/plato/bernie';

        // Define shared dependencies that need to be loaded once
        this.sharedDeps = {
            css: [
                '/share/dnarow.css',
                '/share/icontlbr2.css',
                '/css/msgbar.css',
            ],
            js: [
                '/share/elems.js',
                '/share/dnaweights.js',
                '/share/dnaparser.js',
                '/share/dnarow.js',
                '/share/icontlbr2.js',
                '/js/msgbar.js',
            ]
        };

        this.views = {
            'grpedit': {
                html: '/views/grpedit/grpedit.html',
                css: [
                    '/views/grpedit/grpedit.css',
                ],
                js: [
                    '/views/grpedit/typedown.js',
                    '/views/grpedit/actions.js',
                    '/views/grpedit/grpedit.js'
                ]
            },
            'dnaview': {
                html: '/views/dnaview/dnaview.html',
                css: [
                    '/views/dnaview/dnaview.css'
                ],
                js: [
                    '/views/dnaview/dspmetrics.js',
                    '/views/dnaview/dnaview.js'
                ]
            },
            'crucible': {
                html: '/views/crucible/crucible.html',
                css: [
                    '/views/crucible/crucible.css',
                    '/views/crucible/crucible-investors.css',
                ],
                js: [
                    '/views/crucible/crucible.js',
                    '/views/crucible/crucinv.js'
                ]
            },
            'funds': {
                html: '/views/funds/funds.html',
                css: [
                    '/views/funds/funds.css'
                ],
                js: [
                    '/views/funds/funds.js'
                ]
            },
            'api-docs': {
                url: 'http://localhost:8200/swagger/index.html'
            }
        };

        this.loadedStylesheets = new Set();
        this.loadedScripts = new Set();
    }

    async loadScript(src) {
        if (this.loadedScripts.has(src)) {
            return Promise.resolve();
        }

        return new Promise((resolve, reject) => {
            const script = document.createElement('script');
            script.src = this.baseUrl + src;

            // Extract filename from the full path and remove .js
            // e.g., '/grpedit/typedown.js' becomes 'typedown'
            const filename = src.split('/').pop().replace('.js', '');

            script.onload = () => {
                this.loadedScripts.add(src);
                resolve();
            };
            script.onerror = () => reject(new Error(`Failed to load script: ${src}`));
            document.head.appendChild(script);
        });
    }

    async loadCSS(cssPath) {
        if (this.loadedStylesheets.has(cssPath)) {
            return Promise.resolve();
        }

        return new Promise((resolve, reject) => {
            const link = document.createElement('link');
            link.rel = 'stylesheet';
            link.href = this.baseUrl + cssPath;
            link.onload = () => {
                this.loadedStylesheets.add(cssPath);
                resolve();
            };
            link.onerror = () => reject(new Error(`Failed to load CSS: ${cssPath}`));
            document.head.appendChild(link);
        });
    }

    /**
     * Load all shared CSS and JS files needed by views.
     * If any of the files fail to load, an error is thrown.
     * @throws {Error} if any of the files fail to load
     */
    async loadSharedDependencies() {
        try {
            // Load shared CSS files
            await Promise.all(this.sharedDeps.css.map(css => this.loadCSS(css)));

            // Load shared JS files in sequence (order matters)
            for (const js of this.sharedDeps.js) {
                await this.loadScript(js);
            }

            // Initialize msgbar after we know it's loaded
            if (typeof initMsgbar === 'function') {
                initMsgbar();
            }
        } catch (error) {
            console.error('Error loading shared dependencies:', error);
            throw error;
        }
    }

    async loadViewCSS(viewName) {
        const view = this.views[viewName];
        if (!view || !view.css) return;

        try {
            const cssFiles = Array.isArray(view.css) ? view.css : [view.css];
            await Promise.all(cssFiles.map(cssPath => this.loadCSS(cssPath)));
        } catch (error) {
            console.error('Error loading CSS:', error);
            showMessageBar('Error loading view styles: ' + error.message, 'error');
        }
    }

    async loadView(viewName) {
        const view = this.views[viewName];
        if (!view) return;

        const contentArea = document.getElementById('content-area');

        if (viewName === 'api-docs') {
            window.open(view.url, '_blank');
            return;
        }

        try {
            // Load shared dependencies first if not already loaded
            await this.loadSharedDependencies();

            // Load view-specific CSS
            await this.loadViewCSS(viewName);

            // Load HTML content
            const response = await fetch(this.baseUrl + view.html);
            const html = await response.text();
            const x = document.getElementById('work-area');
            if (x) {
                x.innerHTML = html;
                this.updateActiveView(viewName);
                if (view.js) {
                    const jsFiles = Array.isArray(view.js) ? view.js : [view.js];
                    for (const js of jsFiles) {
                        await this.loadScript(js);
                    }
                    // After loading all scripts, initialize the view
                    const initFnName = `init${viewName.charAt(0).toUpperCase() + viewName.slice(1)}`;
                    console.log(`Initializing view: ${viewName}`);
                    window[initFnName]();
                }
            }
        } catch (error) {
            console.error('Error loading view:', error);
            showMessageBar('Error loading view: ' + error.message, 'error');
        }
    }

    /**
     * Updates the toolbar links to reflect the currently active view.
     * 
     * @param {string} viewName - The name of the view to be marked as active.
     * This function iterates over all navigation links in the toolbar, removing
     * the 'current-view' class from each. It then checks each link's href for 
     * the provided viewName and adds the 'current-view' class to the matching link.
     */
    updateActiveView(viewName) {
        // Remove current-view class from all links
        document.querySelectorAll('.toolbar nav a').forEach(link => {
            link.classList.remove('current-view');
            // Get the view name from the href
            const href = link.getAttribute('href');
            // Check if the href contains the current view name
            if (href && href.includes(viewName)) {
                link.classList.add('current-view');
            }
        });
    }
}

// Create and export a singleton instance
const viewManager = new ViewManager();

export default viewManager;
export { viewManager };